# -*- coding: utf-8 -*-
"""Artificial_Immuse_Networks-Clustering.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/1RGHeyWBLTv4F3Nq3ilUUbcpA4tGGYU8j
"""

import numpy as np
from sklearn.datasets import make_blobs
import matplotlib.pyplot as plt
from scipy.spatial.distance import cdist

class ArtificialImmuneNetwork:
    def __init__(self, n_clusters, suppression_threshold=0.5, clone_factor=0.3, mutation_rate=0.1):
        """
        Initialize the Artificial Immune Network for clustering.

        Parameters:
        - n_clusters: Number of clusters (memory cells)
        - suppression_threshold: Threshold for suppressing similar antibodies
        - clone_factor: Factor determining how many clones to produce
        - mutation_rate: Rate of mutation for cloned antibodies
        """
        self.n_clusters = n_clusters
        self.suppression_threshold = suppression_threshold
        self.clone_factor = clone_factor
        self.mutation_rate = mutation_rate
        self.memory_cells = None

    def _affinity(self, x, y):
        """Calculate affinity (inverse distance) between two points"""
        return 1 / (1 + np.linalg.norm(x - y))

    def _mutate(self, antibody):
        """Mutate an antibody with random noise"""
        mutation = np.random.normal(0, self.mutation_rate, size=antibody.shape)
        return antibody + mutation

    def fit(self, X, max_iter=100):
        """
        Train the Artificial Immune Network on the input data.

        Parameters:
        - X: Input data (n_samples, n_features)
        - max_iter: Maximum number of iterations
        """
        n_samples, n_features = X.shape

        # Step 1: Initialization - randomly select memory cells from data
        indices = np.random.choice(n_samples, self.n_clusters, replace=False)
        self.memory_cells = X[indices].copy()

        for _ in range(max_iter):
            # Step 2: Calculate affinity between all data points and memory cells
            distances = cdist(X, self.memory_cells, 'euclidean')
            affinities = 1 / (1 + distances)

            # Step 3: Assign each data point to the closest memory cell (cluster)
            labels = np.argmax(affinities, axis=1)

            # Step 4: Clone and mutate memory cells based on stimulation
            new_memory_cells = []
            for i in range(self.n_clusters):
                # Get data points assigned to this memory cell
                cluster_points = X[labels == i]
                if len(cluster_points) == 0:
                    new_memory_cells.append(self.memory_cells[i])
                    continue

                # Calculate stimulation (average affinity)
                stimulation = np.mean(affinities[labels == i, i])

                # Number of clones proportional to stimulation
                n_clones = max(1, int(self.clone_factor * stimulation * 100))

                # Clone and mutate
                clones = np.tile(self.memory_cells[i], (n_clones, 1))
                for j in range(n_clones):
                    clones[j] = self._mutate(clones[j])

                # Add original and clones to new population
                new_memory_cells.append(self.memory_cells[i])
                new_memory_cells.extend(clones)

            # Step 5: Select new memory cells with suppression
            new_memory_cells = np.array(new_memory_cells)
            suppressed = set()

            # Calculate pairwise distances between all new memory cells
            mem_distances = cdist(new_memory_cells, new_memory_cells, 'euclidean')

            # Suppress similar memory cells
            for i in range(len(new_memory_cells)):
                if i in suppressed:
                    continue

                # Find similar memory cells
                similar = np.where(mem_distances[i] < self.suppression_threshold)[0]
                similar = [s for s in similar if s != i and s not in suppressed]

                # Keep only the most stimulated (centroid) of similar cells
                if len(similar) > 0:
                    # For simplicity, we'll keep the first one and suppress others
                    suppressed.update(similar)

            # Apply suppression
            final_memory = []
            for i in range(len(new_memory_cells)):
                if i not in suppressed:
                    final_memory.append(new_memory_cells[i])

            # Maintain exactly n_clusters memory cells
            if len(final_memory) > self.n_clusters:
                # Calculate stimulation for each memory cell
                stimulations = []
                for mc in final_memory:
                    dists = cdist(X, [mc], 'euclidean')
                    stimulations.append(np.mean(1 / (1 + dists)))

                # Sort by stimulation and keep top n_clusters
                indices = np.argsort(stimulations)[::-1][:self.n_clusters]
                final_memory = [final_memory[i] for i in indices]
            elif len(final_memory) < self.n_clusters:
                # Add random data points if we don't have enough
                needed = self.n_clusters - len(final_memory)
                indices = np.random.choice(n_samples, needed, replace=False)
                final_memory.extend(X[indices])

            self.memory_cells = np.array(final_memory)

        return self

    def predict(self, X):
        """Predict cluster labels for input data"""
        distances = cdist(X, self.memory_cells, 'euclidean')
        return np.argmin(distances, axis=1)

# Generate sample data
X, y_true = make_blobs(n_samples=300, centers=3, cluster_std=0.8, random_state=42)

# Initialize and train the Artificial Immune Network
ain = ArtificialImmuneNetwork(n_clusters=3)
ain.fit(X)

# Get cluster assignments
y_pred = ain.predict(X)

# Plot the results
plt.figure(figsize=(12, 5))

plt.subplot(1, 2, 1)
plt.scatter(X[:, 0], X[:, 1], c=y_true, cmap='viridis')
plt.title("True Clusters")
plt.scatter(ain.memory_cells[:, 0], ain.memory_cells[:, 1], s=200, c='red', marker='X', label='Memory Cells')
plt.legend()

plt.subplot(1, 2, 2)
plt.scatter(X[:, 0], X[:, 1], c=y_pred, cmap='viridis')
plt.title("AIN Clusters")
plt.scatter(ain.memory_cells[:, 0], ain.memory_cells[:, 1], s=200, c='red', marker='X', label='Memory Cells')
plt.legend()

plt.tight_layout()
plt.show()