#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define Map             "map.txt"       /* file name of city map */
#define NumberOfCity    12              /* number of cities */
#define NumberOfAnt     10              /* number of ants */
#define alpha           0.2             /* pheromone decay factor */
#define beta            2.0             /* tradeoff factor between pheromone and distance */
#define tau0            0.01            /* initial intensity of pheromone */
#define EpisodeLimit    20              /* limit of episode */
#define Route           "route.txt"     /* file name for route map */
#define RAND ((float)rand()/(float)(RAND_MAX+1)) /* Macro for random number between 0 and 1 */

typedef struct {
        float   x;      /* x coordinate */
        float   y;      /* y coordinate */
        } CityType;
typedef struct {
        int     route[NumberOfCity];    /* visiting sequence of cities */
        float   length;                 /* length of route */
        } RouteType;

CityType        city[NumberOfCity];                     /* city array */
float           delta[NumberOfCity][NumberOfCity];      /* distance matrix */
float           eta[NumberOfCity][NumberOfCity];        /* weighted visibility matrix */
float           tau[NumberOfCity][NumberOfCity];        /* pheromone intensity matrix */
RouteType       BestRoute;                              /* shortest route */
RouteType       ant[NumberOfAnt];                       /* ant array */
float           p[NumberOfCity];                        /* path-taking probability array */
int             visited[NumberOfCity];                  /* array for visiting status */
float           delta_tau[NumberOfCity][NumberOfCity];  /* sum of change in tau */

void main(void)
{
FILE*   mapfpr;         /* file pointer for city map */
int     r,s;            /* indices for cities */
int     k;              /* index for ant */
int     episode;        /* index for ant system cycle */
int     step;           /* index for routing step */
float   tmp;            /* temporary variable */
FILE*   routefpr;       /* file pointer for route map */

/* Set random seed */
srand(1);

/* Read city map */
mapfpr=fopen(Map,"r");
for(r=0;r<NumberOfCity;r++)
        fscanf(mapfpr,"%f %f",&(city[r].x),&(city[r].y));
fclose(mapfpr);

/* Evaluate distance matrix */
for(r=0;r<NumberOfCity;r++)
        for(s=0;s<NumberOfCity;s++)
                delta[r][s]=sqrt((city[r].x-city[s].x)*(city[r].x-city[s].x)+(city[r].y-city[s].y)*(city[r].y-city[s].y));

/* Evaluate weighted visibility matrix */
for(r=0;r<NumberOfCity;r++)
        for(s=0;s<NumberOfCity;s++)
                if(r!=s)
                        eta[r][s]=pow((1.0/delta[r][s]),beta);
                else
                        eta[r][s]=0.0;

/* Initialize pheromone on edges */
for(r=0;r<NumberOfCity;r++)
        for(s=0;s<NumberOfCity;s++)
                tau[r][s]=tau0;

/* Initialize best route */
BestRoute.route[0]=0;
BestRoute.length=0.0;
for(r=1;r<NumberOfCity;r++)
        {
        BestRoute.route[r]=r;
        BestRoute.length+=delta[r-1][r];
        }
BestRoute.length+=delta[NumberOfCity-1][0];

/* Repeat ant system cycle for EpisodeLimit times */
for(episode=0;episode<EpisodeLimit;episode++)
        {
        /* Initialize ants' starting city */
        for(k=0;k<NumberOfAnt;k++)
                ant[k].route[0]=RAND*NumberOfCity;

        /* Let all ants proceed for NumberOfCity-1 steps */
        for(step=1;step<NumberOfCity;step++)
                {
                for(k=0;k<NumberOfAnt;k++)
                        {
                        /* Evaluate path-taking probability array for ant k at current time step */
                        r=ant[k].route[step-1];
                        /* Clear visited list of ant k*/
                        for(s=0;s<NumberOfCity;s++)
                                visited[s]=0;
                        /* Mark visited cities of ant k*/
                        for(s=0;s<step;s++)
                                visited[ant[k].route[s]]=1;
                        tmp=0.0;
                        for(s=0;s<NumberOfCity;s++)
                                if(visited[s]==1)
                                        p[s]=0.0;
                                else
                                        {
                                        p[s]=tau[r][s]*eta[r][s];
                                        tmp+=p[s];
                                        }
                        for(s=0;s<NumberOfCity;s++)
                                p[s]/=tmp;
                        /* Probabilistically pick up next edge by roulette wheel selection */
                        tmp=RAND;
                        for(s=0;tmp>p[s];s++)
                                tmp-=p[s];
                        ant[k].route[step]=s;
                        }
                }

        /* Update pheromone intensity */
        /* Reset matrix for sum of change in tau */
        for(r=0;r<NumberOfCity;r++)
                for(s=0;s<NumberOfCity;s++)
                        delta_tau[r][s]=0.0;
        for(k=0;k<NumberOfAnt;k++)
                {
                /* Evaluate route length */
                ant[k].length=0.0;
                for(r=1;r<NumberOfCity;r++)
                        ant[k].length+=delta[ant[k].route[r-1]][ant[k].route[r]];
                ant[k].length+=delta[ant[k].route[NumberOfCity-1]][ant[k].route[0]];
                /* Evaluate contributed delta_tau */
                for(r=1;r<NumberOfCity;r++)
                        delta_tau[ant[k].route[r-1]][ant[k].route[r]]+=1.0/ant[k].length;
                delta_tau[ant[k].route[NumberOfCity-1]][ant[k].route[0]]+=1.0/ant[k].length;
                /* Update best route */
                if(ant[k].length<BestRoute.length)
                        {
                        BestRoute.length=ant[k].length;
                        for(r=0;r<NumberOfCity;r++)
                                BestRoute.route[r]=ant[k].route[r];
                        }
                }
        /* Update pheromone matrix */
        for(r=0;r<NumberOfCity;r++)
                for(s=0;s<NumberOfCity;s++)
                        tau[r][s]=(1.0-alpha)*tau[r][s]+delta_tau[r][s];
        printf("%d %f\n",episode,BestRoute.length);
        }

/* Write route map */
routefpr=fopen(Route,"w");
for(r=0;r<NumberOfCity;r++)
        fprintf(routefpr,"%f %f\n",city[BestRoute.route[r]].x,city[BestRoute.route[r]].y);
fclose(routefpr);

/* Report pheromone intensity */
/*
printf("%d\n",episode);
for(r=0;r<NumberOfCity;r++)
	for(s=r+1;s<NumberOfCity;s++)
		printf("%f %f %f %f %f\n",city[r].x,city[r].y,city[s].x,city[s].y,tau[r][s]);
*/
}
